<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>qPS4 Score Calculator</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: #0f172a;
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 650px;
            margin: 0 auto;
            background: #1e293b;
            border-radius: 16px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.5);
            overflow: hidden;
            border: 1px solid #334155;
        }
        
        .header {
            background: #334155;
            color: #e2e8f0;
            padding: 30px 20px;
            text-align: center;
            border-bottom: 2px solid #475569;
        }
        
        .header h1 {
            font-size: 28px;
            margin-bottom: 8px;
        }
        
        .header p {
            opacity: 0.9;
            font-size: 14px;
        }
        
        .header-controls {
            margin-top: 16px;
            display: flex;
            gap: 12px;
            justify-content: center;
            flex-wrap: wrap;
        }
        
        .lang-btn, .download-btn {
            padding: 8px 16px;
            background: #475569;
            color: #e2e8f0;
            border: none;
            border-radius: 6px;
            font-size: 14px;
            cursor: pointer;
            transition: all 0.2s;
        }
        
        .lang-btn:hover, .download-btn:hover {
            background: #64748b;
        }
        
        .lang-btn.active {
            background: #3b82f6;
        }
        
        .form-section {
            padding: 30px 20px;
        }
        
        .section-header {
            background: #334155;
            padding: 12px 16px;
            margin: 0 -20px 20px -20px;
            font-weight: 700;
            color: #e2e8f0;
            border-left: 4px solid #3b82f6;
        }
        
        .form-group {
            margin-bottom: 24px;
        }
        
        label {
            display: block;
            font-weight: 600;
            color: #cbd5e1;
            margin-bottom: 8px;
            font-size: 15px;
        }
        
        input[type="number"],
        select {
            width: 100%;
            padding: 12px 16px;
            border: 2px solid #475569;
            border-radius: 8px;
            font-size: 16px;
            transition: all 0.3s;
            background: #0f172a;
            color: #e2e8f0;
        }
        
        input[type="number"]:focus,
        select:focus {
            outline: none;
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.2);
        }
        
        .unit {
            font-size: 13px;
            color: #94a3b8;
            margin-top: 4px;
        }
        
        .btn {
            width: 100%;
            padding: 16px;
            background: #3b82f6;
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s;
        }
        
        .btn:hover {
            background: #2563eb;
            transform: translateY(-2px);
        }
        
        .btn:active {
            transform: translateY(0);
        }
        
        .result {
            margin-top: 30px;
            padding: 24px;
            background: #0f172a;
            border-radius: 12px;
            border-left: 4px solid #3b82f6;
        }
        
        .result h2 {
            color: #e2e8f0;
            margin-bottom: 16px;
            font-size: 20px;
        }
        
        .score-display {
            font-size: 48px;
            font-weight: 700;
            color: #3b82f6;
            margin: 16px 0;
        }
        
        .risk-level {
            display: inline-block;
            padding: 8px 16px;
            border-radius: 6px;
            font-weight: 600;
            margin: 12px 0;
        }
        
        .risk-low {
            background: #065f46;
            color: #d1fae5;
        }
        
        .risk-moderate {
            background: #92400e;
            color: #fef3c7;
        }
        
        .risk-high {
            background: #991b1b;
            color: #fecaca;
        }
        
        .calculation-steps {
            margin-top: 20px;
            padding: 16px;
            background: #1e293b;
            border-radius: 8px;
            border: 1px solid #334155;
        }
        
        .calculation-steps h3 {
            color: #e2e8f0;
            margin-bottom: 12px;
            font-size: 16px;
        }
        
        .step {
            padding: 10px 0;
            border-bottom: 1px solid #334155;
            font-size: 14px;
            color: #cbd5e1;
        }
        
        .step:last-child {
            border-bottom: none;
        }
        
        .step-value {
            float: right;
            font-weight: 600;
            color: #3b82f6;
        }
        
        .subscore {
            background: #1e3a5f;
            padding: 12px;
            border-radius: 6px;
            margin: 10px 0;
            border-left: 3px solid #3b82f6;
            color: #e2e8f0;
        }
        
        .info-box {
            background: #1e3a5f;
            border-left: 4px solid #3b82f6;
            padding: 16px;
            border-radius: 8px;
            margin-bottom: 24px;
            font-size: 14px;
            color: #cbd5e1;
        }
        
        .disclaimer {
            background: #1e293b;
            border: 1px solid #334155;
            border-radius: 8px;
            padding: 16px;
            margin-bottom: 24px;
            font-size: 13px;
            color: #94a3b8;
        }
        
        .disclaimer strong {
            color: #cbd5e1;
            display: block;
            margin-bottom: 8px;
        }
        
        .wip-banner {
            background: #991b1b;
            color: #fecaca;
            padding: 16px 20px;
            text-align: center;
            font-weight: 600;
            font-size: 14px;
            border-bottom: 2px solid #7f1d1d;
        }
        
        .wip-banner strong {
            display: block;
            font-size: 16px;
            margin-bottom: 4px;
        }
        
        .source-info {
            background: #1e3a5f;
            border-left: 4px solid #3b82f6;
            padding: 16px;
            border-radius: 8px;
            margin-bottom: 24px;
            font-size: 13px;
            color: #cbd5e1;
        }
        
        .source-info strong {
            color: #e2e8f0;
            display: block;
            margin-bottom: 8px;
        }
        
        .source-info a {
            color: #60a5fa;
            text-decoration: none;
        }
        
        .source-info a:hover {
            text-decoration: underline;
        }
        
        .footer {
            background: #334155;
            padding: 20px;
            text-align: center;
            color: #94a3b8;
            font-size: 13px;
            border-top: 1px solid #475569;
        }
        
        .footer a {
            color: #3b82f6;
            text-decoration: none;
        }
        
        .footer a:hover {
            text-decoration: underline;
        }
        
        @media (max-width: 480px) {
            .header h1 {
                font-size: 24px;
            }
            
            .score-display {
                font-size: 36px;
            }
            
            .header-controls {
                flex-direction: column;
            }
            
            .lang-btn, .download-btn {
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <?php
    // Handle download BEFORE any output
    if (isset($_GET['download'])) {
        $filename = 'qps4_calculator.php';
        $content = file_get_contents(__FILE__);
        
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen($content));
        header('Cache-Control: no-cache, must-revalidate');
        header('Pragma: no-cache');
        header('Expires: 0');
        
        echo $content;
        exit;
    }
    
    $browser_lang = substr($_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? 'en', 0, 2);
    $lang = $_GET['lang'] ?? ($_COOKIE['lang'] ?? $browser_lang);
    
    if (isset($_GET['lang'])) {
        setcookie('lang', $_GET['lang'], time() + (86400 * 365), "/");
    }
    
    if ($lang !== 'de') {
        $lang = 'en';
    }
    
    $t = [
        'en' => [
            'title' => 'qPS4 Score Calculator',
            'subtitle' => 'quick Pediatric Septic Shock Score',
            'disclaimer_title' => 'Privacy & Data Protection:',
            'disclaimer_text' => 'No patient data is stored on the server. We only collect anonymous statistics on how often this site is visited. All calculations are performed in real-time and no personal health information is retained.',
            'download' => 'Download Source Code',
            'mental_status' => 'Mental Status',
            'select' => 'Select...',
            'normal' => 'Normal / Alert',
            'altered' => 'Altered / Impaired',
            'resp_rate' => 'Respiratory Rate',
            'age_group' => 'Patient Age Group',
            'breaths' => 'breaths per minute',
            'heart_rate' => 'Heart Rate',
            'bpm' => 'beats per minute',
            'map' => 'Mean Arterial Pressure (MAP)',
            'mmhg' => 'mmHg',
            'temp' => 'Temperature',
            'celsius' => 'degrees C (Celsius)',
            'crt' => 'Capillary Refill Time',
            'crt_normal' => 'Normal (≤2 seconds)',
            'crt_prolonged' => 'Prolonged (>2 seconds)',
            'calculate' => 'Calculate qPS4 Score',
            'result' => 'qPS4 Score Result',
            'steps' => 'Detailed Calculation Steps:',
            'final' => 'Final qPS4 Score:',
            'risk_low' => 'Low Risk',
            'risk_lowmod' => 'Low-Moderate Risk',
            'risk_mod' => 'Moderate Risk',
            'risk_high' => 'High Risk',
            'risk_veryhigh' => 'Very High Risk',
            'interp0' => 'Low probability of septic shock. Continue routine monitoring.',
            'interp1' => 'Low to moderate probability of septic shock. Continue monitoring with clinical assessment.',
            'interp2' => 'Moderate probability of septic shock. Increased monitoring and clinical evaluation recommended.',
            'interp3' => 'High probability of septic shock. Urgent evaluation and intervention indicated.',
            'interp4' => 'Very high probability of septic shock. Immediate aggressive intervention required.',
            'license' => 'License: MIT',
            'license_text' => 'This calculator is open source and released under the MIT License. You are free to use, modify, and distribute this software.',
            'imprint' => 'Imprint',
            'contact' => 'Contact',
            'wip_title' => 'WORK IN PROGRESS',
            'wip_text' => 'This calculator is currently under development. Please verify all results independently.',
            'source_title' => 'Clinical Source',
            'source_text' => 'This calculator is based on the qPS4 (quick Pediatric Septic Shock Score) as described in:',
            'source_citation' => 'Horeczko T, et al. Pediatric Septic Shock: Quick Recognition and Early Resuscitation. Pediatr Emerg Care. 2013.',
        ],
        'de' => [
            'title' => 'qPS4 Score Rechner',
            'subtitle' => 'quick Pediatric Septic Shock Score',
            'disclaimer_title' => 'Datenschutz & Privatsphäre:',
            'disclaimer_text' => 'Es werden keine Patientendaten auf dem Server gespeichert. Wir erfassen nur anonyme Statistiken über die Häufigkeit der Seitenbesuche. Alle Berechnungen erfolgen in Echtzeit und es werden keine persönlichen Gesundheitsinformationen gespeichert.',
            'download' => 'Quellcode Herunterladen',
            'mental_status' => 'Bewusstseinszustand',
            'select' => 'Auswählen...',
            'normal' => 'Normal / Wach',
            'altered' => 'Verändert / Beeinträchtigt',
            'resp_rate' => 'Atemfrequenz',
            'age_group' => 'Altersgruppe des Patienten',
            'breaths' => 'Atemzüge pro Minute',
            'heart_rate' => 'Herzfrequenz',
            'bpm' => 'Schläge pro Minute',
            'map' => 'Mittlerer arterieller Druck (MAP)',
            'mmhg' => 'mmHg',
            'temp' => 'Temperatur',
            'celsius' => 'Grad C (Celsius)',
            'crt' => 'Rekapillarisierungszeit',
            'crt_normal' => 'Normal (≤2 Sekunden)',
            'crt_prolonged' => 'Verlängert (>2 Sekunden)',
            'calculate' => 'qPS4 Score Berechnen',
            'result' => 'qPS4 Score Ergebnis',
            'steps' => 'Detaillierte Berechnungsschritte:',
            'final' => 'Finaler qPS4 Score:',
            'risk_low' => 'Niedriges Risiko',
            'risk_lowmod' => 'Niedrig-Mittleres Risiko',
            'risk_mod' => 'Mittleres Risiko',
            'risk_high' => 'Hohes Risiko',
            'risk_veryhigh' => 'Sehr Hohes Risiko',
            'interp0' => 'Geringe Wahrscheinlichkeit eines septischen Schocks. Routineüberwachung fortsetzen.',
            'interp1' => 'Niedrige bis mittlere Wahrscheinlichkeit eines septischen Schocks. Überwachung mit klinischer Beurteilung fortsetzen.',
            'interp2' => 'Mittlere Wahrscheinlichkeit eines septischen Schocks. Verstärkte Überwachung und klinische Bewertung empfohlen.',
            'interp3' => 'Hohe Wahrscheinlichkeit eines septischen Schocks. Dringende Evaluation und Intervention angezeigt.',
            'interp4' => 'Sehr hohe Wahrscheinlichkeit eines septischen Schocks. Sofortige aggressive Intervention erforderlich.',
            'license' => 'Lizenz: MIT',
            'license_text' => 'Dieser Rechner ist Open Source und unter der MIT-Lizenz veröffentlicht. Sie dürfen diese Software frei verwenden, modifizieren und verteilen.',
            'imprint' => 'Impressum',
            'contact' => 'Kontakt',
            'wip_title' => 'IN ENTWICKLUNG',
            'wip_text' => 'Dieser Rechner befindet sich derzeit in der Entwicklung. Bitte überprüfen Sie alle Ergebnisse unabhängig.',
            'source_title' => 'Klinische Quelle',
            'source_text' => 'Dieser Rechner basiert auf dem qPS4 (quick Pediatric Septic Shock Score) wie beschrieben in:',
            'source_citation' => 'Horeczko T, et al. Pediatric Septic Shock: Quick Recognition and Early Resuscitation. Pediatr Emerg Care. 2013.',
        ]
    ][$lang];
    ?>
    
    <div class="wip-banner">
        <strong><?php echo $t['wip_title']; ?></strong>
        <?php echo $t['wip_text']; ?>
    </div>
    
    <div class="container">
        <div class="header">
            <h1><?php echo $t['title']; ?></h1>
            <p><?php echo $t['subtitle']; ?></p>
            <div class="header-controls">
                <button class="lang-btn <?php echo $lang === 'en' ? 'active' : ''; ?>" onclick="window.location.href='?lang=en'">English</button>
                <button class="lang-btn <?php echo $lang === 'de' ? 'active' : ''; ?>" onclick="window.location.href='?lang=de'">Deutsch</button>
                <a href="qps4_calculator.zip" target="_blank" class="download-btn" style="display: inline-block; text-decoration: none; line-height: 1.5;"><?php echo $t['download']; ?></a>
            </div>
        </div>
        
        <div class="form-section">
            <div class="source-info">
                <strong><?php echo $t['source_title']; ?>:</strong>
                <?php echo $t['source_text']; ?><br>
                <em><?php echo $t['source_citation']; ?></em>
            </div>
            
            <div class="disclaimer">
                <strong><?php echo $t['disclaimer_title']; ?></strong>
                <?php echo $t['disclaimer_text']; ?>
            </div>
            
            <form method="POST">
                <div class="section-header">1. <?php echo $t['mental_status']; ?></div>
                <div class="form-group">
                    <label><?php echo $t['mental_status']; ?></label>
                    <select name="mental" required>
                        <option value=""><?php echo $t['select']; ?></option>
                        <option value="0"><?php echo $t['normal']; ?></option>
                        <option value="1"><?php echo $t['altered']; ?></option>
                    </select>
                </div>
                
                <div class="section-header">2. <?php echo $t['resp_rate']; ?></div>
                <div class="form-group">
                    <label><?php echo $t['age_group']; ?></label>
                    <select name="age" required>
                        <option value=""><?php echo $t['select']; ?></option>
                        <option value="0-1m">0-1 <?php echo $lang === 'de' ? 'Monat' : 'month'; ?></option>
                        <option value="1-3m">1-3 <?php echo $lang === 'de' ? 'Monate' : 'months'; ?></option>
                        <option value="3-6m">3-6 <?php echo $lang === 'de' ? 'Monate' : 'months'; ?></option>
                        <option value="6-12m">6-12 <?php echo $lang === 'de' ? 'Monate' : 'months'; ?></option>
                        <option value="1-2y">1-2 <?php echo $lang === 'de' ? 'Jahre' : 'years'; ?></option>
                        <option value="2-5y">2-5 <?php echo $lang === 'de' ? 'Jahre' : 'years'; ?></option>
                        <option value="5-12y">5-12 <?php echo $lang === 'de' ? 'Jahre' : 'years'; ?></option>
                        <option value="12-18y">12-18 <?php echo $lang === 'de' ? 'Jahre' : 'years'; ?></option>
                    </select>
                </div>
                <div class="form-group">
                    <label><?php echo $t['resp_rate']; ?></label>
                    <input type="number" name="rr" required>
                    <div class="unit"><?php echo $t['breaths']; ?></div>
                </div>
                
                <div class="section-header">3. TAMSI</div>
                <div class="form-group">
                    <label><?php echo $t['heart_rate']; ?></label>
                    <input type="number" name="hr" required>
                    <div class="unit"><?php echo $t['bpm']; ?></div>
                </div>
                <div class="form-group">
                    <label><?php echo $t['map']; ?></label>
                    <input type="number" name="map" step="0.1" required>
                    <div class="unit"><?php echo $t['mmhg']; ?></div>
                </div>
                <div class="form-group">
                    <label><?php echo $t['temp']; ?></label>
                    <input type="number" name="temp" step="0.1" required>
                    <div class="unit"><?php echo $t['celsius']; ?></div>
                </div>
                
                <div class="section-header">4. <?php echo $t['crt']; ?></div>
                <div class="form-group">
                    <label><?php echo $t['crt']; ?></label>
                    <select name="crt" required>
                        <option value=""><?php echo $t['select']; ?></option>
                        <option value="0"><?php echo $t['crt_normal']; ?></option>
                        <option value="1"><?php echo $t['crt_prolonged']; ?></option>
                    </select>
                </div>
                
                <button type="submit" class="btn"><?php echo $t['calculate']; ?></button>
            </form>
            
            <?php
            if ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $mental = intval($_POST['mental']);
                $age = $_POST['age'];
                $rr = intval($_POST['rr']);
                $hr = intval($_POST['hr']);
                $map = floatval($_POST['map']);
                $temp = floatval($_POST['temp']);
                $crt = intval($_POST['crt']);
                
                $steps = [];
                $score = 0;
                
                // Mental Status
                if ($mental == 0) {
                    $steps[] = ($lang === 'de' ? 'Bewusstseinszustand: Normal/Wach' : 'Mental Status: Normal/Alert') . ' → 0';
                } else {
                    $steps[] = ($lang === 'de' ? 'Bewusstseinszustand: Verändert' : 'Mental Status: Altered') . ' → 1';
                }
                $score += $mental;
                
                // Respiratory Rate
                $rr_thresholds = ['0-1m'=>60, '1-3m'=>50, '3-6m'=>45, '6-12m'=>40, '1-2y'=>35, '2-5y'=>30, '5-12y'=>25, '12-18y'=>20];
                $rr_thresh = $rr_thresholds[$age];
                $rr_score = ($rr > $rr_thresh) ? 1 : 0;
                $steps[] = ($lang === 'de' ? 'Atemfrequenz' : 'Respiratory Rate') . ": {$rr} (>{$rr_thresh}) → {$rr_score}";
                $score += $rr_score;
                
                // TAMSI
                $steps[] = "<div class='subscore'>TAMSI " . ($lang === 'de' ? 'Berechnung' : 'Calculation') . ":</div>";
                $tamsi = 0;
                
                $hr_thresholds = ['0-1m'=>180, '1-3m'=>180, '3-6m'=>180, '6-12m'=>170, '1-2y'=>160, '2-5y'=>150, '5-12y'=>130, '12-18y'=>110];
                $hr_thresh = $hr_thresholds[$age];
                $hr_score = ($hr > $hr_thresh) ? 1 : 0;
                $steps[] = "  • " . ($lang === 'de' ? 'Herzfrequenz' : 'Heart Rate') . ": {$hr} (>{$hr_thresh}) → {$hr_score}";
                $tamsi += $hr_score;
                
                $map_thresholds = ['0-1m'=>45, '1-3m'=>50, '3-6m'=>55, '6-12m'=>55, '1-2y'=>55, '2-5y'=>60, '5-12y'=>65, '12-18y'=>70];
                $map_thresh = $map_thresholds[$age];
                $map_score = ($map < $map_thresh) ? 1 : 0;
                $steps[] = "  • MAP: {$map} (<{$map_thresh}) → {$map_score}";
                $tamsi += $map_score;
                
                $temp_score = ($temp < 36 || $temp > 38.5) ? 1 : 0;
                $steps[] = "  • " . ($lang === 'de' ? 'Temperatur' : 'Temperature') . ": {$temp}°C → {$temp_score}";
                $tamsi += $temp_score;
                
                $steps[] = "<div class='subscore'>TAMSI: {$tamsi}/3</div>";
                $tamsi_contrib = ($tamsi >= 2) ? 1 : 0;
                $steps[] = "TAMSI → qPS4: {$tamsi_contrib}";
                $score += $tamsi_contrib;
                
                // CRT
                $steps[] = ($lang === 'de' ? 'Rekapillarisierungszeit' : 'Capillary Refill Time') . ": " . ($crt == 0 ? ($lang === 'de' ? 'Normal' : 'Normal') : ($lang === 'de' ? 'Verlängert' : 'Prolonged')) . " → {$crt}";
                $score += $crt;
                
                $risks = ['risk_low', 'risk_low', 'risk_moderate', 'risk_high', 'risk_high'];
                $risk_labels = [$t['risk_low'], $t['risk_lowmod'], $t['risk_mod'], $t['risk_high'], $t['risk_veryhigh']];
                $interps = [$t['interp0'], $t['interp1'], $t['interp2'], $t['interp3'], $t['interp4']];
                
                echo '<div class="result">';
                echo '<h2>' . $t['result'] . '</h2>';
                echo '<div class="score-display">' . $score . ' / 4</div>';
                echo '<div class="risk-level ' . $risks[$score] . '">' . $risk_labels[$score] . '</div>';
                echo '<p style="margin-top: 12px; color: #cbd5e1;">' . $interps[$score] . '</p>';
                echo '<div class="calculation-steps"><h3>' . $t['steps'] . '</h3>';
                foreach ($steps as $step) {
                    echo '<div class="step">' . $step . '</div>';
                }
                echo '<div class="step" style="font-weight: 700; color: #e2e8f0; border-top: 2px solid #3b82f6; margin-top: 12px; padding-top: 12px;">';
                echo $t['final'] . ' <span class="step-value">' . $score . ' / 4</span></div>';
                echo '</div></div>';
            }
            ?>
        </div>
        
        <div class="footer">
            <strong><?php echo $t['license']; ?></strong><br>
            <?php echo $t['license_text']; ?><br><br>
            <a href="https://opensource.org/licenses/MIT" target="_blank">https://opensource.org/licenses/MIT</a>
            
            <div style="margin-top: 20px; padding-top: 20px; border-top: 1px solid #475569;">
                <strong><?php echo $t['imprint']; ?></strong><br>
                <?php echo $t['contact']; ?>: <a href="mailto:your-email@example.com">your-email@example.com</a>
            </div>
        </div>
    </div>
</body>
</html>
